/*****************************************************************

    This file includes the utility routines used in the program

******************************************************************/
#define UTIL_SOURCE_C_C
#include "util.h"

float Find_Power_For_Tail_Factor(int bin_num, float tail_factor, 
				 float epsilon) 
{
  double start, end, mid;
  double center;
  double tail_bin_val, center_bin_val;
  double func_val;
  if (tail_factor <= 1.0000000001) return 0.0;
  center = (float)(bin_num-1)/2.0;
  tail_bin_val = center;
  center_bin_val = center-(int)((bin_num-1)/2.0);
  if (center_bin_val < 0.50)
    center_bin_val = 1/2.0;
  mexPrintf("Tail bin: %10.8lf center bin: %10.8lf ",
	 tail_bin_val, center_bin_val);
  mexPrintf("factor: %8.6f epsilon: %8.6f\n", tail_factor, epsilon);
	 
  start = 0.0000;
  end = tail_factor;
 
  do {
    mid = (start+end)/2.0;
    func_val = pow(tail_bin_val,mid)+1.0 - 
      tail_factor * (pow(center_bin_val,mid)+1.0);
    mexPrintf("Start=%8.6lf mid=%8.6lf end=%8.6lf fun = %12.10lf %12.10f\n",
	   start, mid, end, func_val, epsilon);
    
    if (fabs(func_val) < epsilon) break;
    if (func_val >0) end = mid;
    else start = mid;
  } while(1);
  mexPrintf("mid=%12.10lf fun = %12.10lf %12.10f\n",
	 mid, func_val, epsilon);
  return (float)mid;
}

/*********************************************/
int  Round(float x)
{ 
  if(x>0) 
    return( (int)(x+0.5) );
  return( (int)(x-0.5) );
}

/**********************************************/
/*             return random [0,1]            */
/**********************************************/
float Unit_Random(void)
{
  float r;
  
 /*  return(rand()/2147483647.0); */   /* 2**31-1 */

  r=(rand()%32767)/32767;
  while(r==0 || r==1)  r=(rand()%32767)/32767.0;
  /*if (r < 0 || r >1) {
    mexPrintf(" r %8.6f is not between 0 and 1.\n", r);
  }
  */
  return(r); 
}

MAT_VECT Create_3D_Matrix(int height,int width, int length)
{
  MAT_VECT p;
  int i;
  mexPrintf("Creating a vector image with size %d x %d x %d ... ",
	 height, width, length);
  mexEvalString("drawnow;");
  p = (MAT_VECT)malloc(sizeof(MAT)*height);
  for (i=0; i< height; i++) {
    if ( ((20*i)%(height))==0) {
      mexPrintf("-");
      mexEvalString("drawnow;");
    }
    p[i] = Create_Matrix(width, length);
  }
  mexPrintf(" Done!\n");
  return p;
}

void Free_3D_Matrix(MAT_VECT a3Dmat, int height, int width)
{
  int i;
  for (i=0; i < height; i++) 
    Free_Matrix(a3Dmat[i], width);
  free(a3Dmat);
}

/*******************************************************
         allocate the space for matrix
*******************************************************/
MAT  Create_Matrix(int height,int width)
{
  MAT  p;
  int  i;
  
  p=(float **)malloc(height*sizeof(float *));
  for(i=0; i<height; i++) {
    p[i]=(float *)malloc(width*sizeof(float));
    if (p[i] == NULL) {
      mexPrintf("Cannot create row %d for image %d x %d\n",
	     i, height, width);
      exit(-1);
    }
  }
  return(p);
}

void Free_Matrix(MAT anmat, int height)
{
  int i;
  for (i=0; i<height; i++) 
    free(anmat[i]);
  free(anmat);
}

/*****************************************************/
MAT_I  Create_Int_Matrix(int height,int width)
{
  MAT_I  p;
  int  i;
  
  p=(int **)malloc(height*sizeof(int *));
  for(i=0; i<height; i++) {
    p[i]=(int *)malloc(width*sizeof(int));
    if (p[i] == NULL) {
      mexPrintf("Cannot create row %d for image %d x %d\n",
	     i, height, width);
      exit(-1);
    }
  }
  return(p);
}

void Free_Int_Matrix(MAT_I anImat, int height)
{
  int i;
  for (i=0; i<height; i++) 
    free(anImat[i]);
  free(anImat);
}

/****************************************************/
ARRAY Create_Array(int length)
{
  return( (float *)malloc(length*sizeof(float)) );
}

void Free_Array(ARRAY anarrary)
{
  free(anarrary);
}

int Free_Double_Matrix(DMAT aDmat,int size)
{
  int i;
  for (i=0; i<size; i++)
    free(aDmat[i]);
  free(aDmat);
  return 0;
}

/*******************************************************
         allocate the space for matrix
*******************************************************/
DMAT  Create_Double_Matrix(int height, int width)
{
  DMAT  p;
  int  i;
  
  p=(double **)malloc(height*sizeof(double *));
  for(i=0; i<height; i++)
    p[i]=(double *)malloc(width*sizeof(double));
  
  return(p);
}


/************************************************************/
int Copy_Double_Matrix(DMAT from, DMAT to, int size)
{
  int i,j;
  
  for(i=0; i<size; i++)
    for(j=0; j<size; j++)
      to[i][j]=from[i][j];
}


/************************************************
   initialize the pyramid based on the bottom  matrix
************************************************/
MAT_I Subsample(MAT_I mat0, int row_size, int col_size)
{
  int   i, j;
  MAT_I  mat;

  mat=Create_Int_Matrix(row_size, col_size);
  
  for(i=0; i<row_size; i++)
    for(j=0; j<col_size; j++)
      mat[i][j]=mat0[2*i][2*j];
  
  return(mat);
}



/****************************************************/
void Normalize_Int_Matrix(MAT_I mat, 
			  int height,int width,int low,int high)
{
  int i,j;
  float min, max, ratio;
  
  min=1000; max=0;
  for(i=0; i<height; i++)
   for(j=0; j<width; j++)
     if(mat[i][j]<min) min=mat[i][j];
     else if (mat[i][j]>max) max=mat[i][j];
  if (min == max)
    max = min +1;

  ratio= ((float)(high-low-1.0))/((float)(max-min));
  for(i=0; i<height; i++)
    for(j=0; j<width; j++)
      mat[i][j]=(int)( (mat[i][j]-min)*ratio+low );
}

/****************************************************/
int Normalize_Image_Matrix(MAT_I mat, int height, int width,
			   int graylevel, float tailPerc)
{
  int i,j;
  float min, max, ratio;
  int  adjustTail;
  int  minT, maxT;
  float unit;
  int   pVal;
  int   trials;
  mexPrintf("Renormalizing image to %d gray levels with %6.2f %% ",
	 graylevel, (float)(tailPerc *100.));
  mexPrintf("percentage.\n");
	 
  min=mat[0][0]; max=mat[0][0];
  for(i=0; i<height; i++) {
    for(j=0; j<width; j++) {
      if(mat[i][j]<min) min=mat[i][j];
      else if (mat[i][j]>max) max=mat[i][j];
    }
  }
  trials = 0;
  do {
    if ((trials %10) ==0) {
      mexPrintf("Trial %d => Adjusting: now min = %6.2f max = %6.2f\n",
	     trials, min,max);
    }
    adjustTail = 0;
    unit = (max-min)/graylevel;
    minT =0; maxT =0;
    for(i=0; i<height; i++)
      for(j=0; j<width; j++) {
        if (mat[i][j] < (min+unit))
          minT++;
        if (mat[i][j] > (max-unit))
          maxT++;
      }
    if (minT < (height*width * tailPerc)) {
      min += unit*0.05;
      adjustTail = 1;
    }
    if (maxT < (height*width * tailPerc)) {
      max -= unit*0.05;
      adjustTail = 1;
    }
    trials ++;
  } while(adjustTail && trials < 1000);
  mexPrintf("Adjusted min = %8.6f max = %8.6f\n", min, max);
  ratio=graylevel/(max-min);
  for(i=0; i<height; i++) {
    for(j=0; j<width; j++) {
      pVal = (int)( (mat[i][j]-min)*ratio );
      if (pVal < 0) pVal = 0;
      if (pVal >= graylevel) pVal = graylevel-1;
      mat[i][j]= pVal;
    }
  }
  
  return;
}

/****************************************************/
int Reverse_Normalize_Image_Matrix(MAT_I mat, int height, int width,
				   int graylevel,
				   float min, float max)
{
  int i,j;
  float ratio;
  float unit;
  int   pVal;
  mexPrintf("Reversing normalization image to 0 => %6.2f and %d => %6.2f.\n",
	 min, graylevel-1, max);
	 
  ratio=(max-min)/((graylevel-1)*(256/graylevel));
  for(i=0; i<height; i++) {
    for(j=0; j<width; j++) {
      pVal = (int)( (mat[i][j])*ratio +min);
      mat[i][j]= pVal;
    }
  }
  
  return;
}


# define DELTA 1e-9     /* allowable minimum positive */



/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*           Sweep                 */
/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

/* 
  Input: A -- (n x n) matrix.  
         n -- the dimensions of A. 
         k -- to sweep with respect to the pivotal element A[k][k]
              k=0,..,n-1.                                             
         flag = 1 for Sweep, = -1 for reverse Sweep.  
  return: A -- sweeped matrix. 
          0 -- succesful. 
*/

int Sweep(DMAT A, int n, int k,  int flag)
{
  static double *u=NULL;
  static int    m=0;
  int i, j;
  double c;
  
  if (m<n)  {
    if (u!=NULL) { free(u); }
    if((u=(double*)calloc(n,sizeof(double)))==NULL) {
      mexPrintf("Sweep: no memory available!\n"); 
      exit(1); 
    }  
    m=n;
  }
  
  if ((c=A[k][k])*flag<DELTA) {
    mexPrintf("Sweep %d: nonpositive matrix!\n", k);
    return 1;
  }
  
  c=1.0/c;
  for (j=0; j<=k; j++)  {
    u[j]=A[k][j];
    A[k][j]=0.0;
  }
  for (i=k+1; i<n; i++) {
    u[i]=A[i][k];
    A[i][k]=0.0;
  }
 
  u[k]=-1.0*flag;
  for (i=0; i<n; i++)  {
    for (j=0; j<=i; j++) {
      A[i][j] -= c*u[i]*u[j];
      A[j][i] = A[i][j];
    }
    
  }
  return 0;
}

/************************************************
   initialize the matrix as random noise
************************************************/
MAT_I Create_Noise_Matrix(int height,int width, int minVal, int maxVal)
{
  int i, j;
  MAT_I mat;
  
  mat=Create_Int_Matrix(height, width);
  
  for(i=0; i<height; i++)
    for(j=0; j<width; j++) {
      mat[i][j]=minVal+(int)(Unit_Random()*(maxVal-minVal+1));
      if (mat[i][j] > maxVal)
	mat[i][j] = maxVal;
    }
  return(mat);
}

/************************************************
   initialize the matrix as random noise
************************************************/
MAT_I Create_Const_Matrix(int height,int width, int cVal)
{
  int i, j;
  MAT_I mat;
   
   mat=Create_Int_Matrix(height, width);

   for(i=0; i<height; i++)
     for(j=0; j<width; j++)
       mat[i][j]=cVal;

   return(mat);
}
