/***********************************************************

   This file inludes all operators on the data structure : kernel

     wid, ht                (must be odd numbers)
     * * * * *
     * * * * *
     * * * * *

***********************************************************/
#define  KERNEL_SOURCE_C_C

#include "kernel.h"

/*************************************************

     Read kernel from file to filter

*************************************************/

void Read_Kernel(FILTER *filter, char *name, int layer_no)
     /*FILTER *filter: read to this filter which uses this kernel */
     /*char *name: file name storing the kernel */
     /* int layer_no; */ 
{
  FILE *kernel;
  float   *p;
  int row, col, i,j, count, scale;
  static int fold[4]={1,2,4,8};
  
  kernel=fopen(name, "r");
  if(kernel==NULL) {
    printf(" open kernel file %s error\n", name);
    exit(-1);
  }

  fscanf(kernel, "%d%d", &row, &col);
  if( (row%2==0) || (col%2==0) ) {
    printf("error: kernel size must be odd number!\n");
    exit(-1);
  }
  
  filter->size=row*col;  scale=fold[layer_no];
  
  filter->ht=row/2 *scale; filter->wid=col/2 *scale;
  
  filter->kernel=Create_Array(filter->size);
  
  p=filter->kernel;
  
  for(i=0; i<filter->size; i++)
    fscanf(kernel, "%f", p++);
  strcpy(filter->name, name);
  printf("  Reading kernel %d  from %s: ",  filter->id-1,
	 name);
  printf("size %d x %d, expanded to %d x %d\n", 
	 row, col,
	 filter->ht*2+1, filter->wid*2+1);
  
  if(filter->ht!=0 || filter->wid!=0)
    Normalize_Kernel(filter);
  
  filter->rlist=(int *)malloc(filter->size*sizeof(int));
  filter->clist=(int *)malloc(filter->size*sizeof(int));
  count=0;
  for(i=-filter->ht; i<=filter->ht; i+=scale)
    for(j=-filter->wid; j<=filter->wid; j+=scale) {
      filter->rlist[count]=i;
      filter->clist[count]=j;
      count++;
    }
  fclose(kernel);
  /* Create the filter mark array */
  filter->mark = Create_Array(filter->bin_num);
  filter->hist = Create_Array(filter->bin_num);
  filter->target_hist = Create_Array(filter->bin_num);
  filter->lambda = Create_Array(filter->bin_num);
  for (i=0; i < filter->bin_num; i++) {
    filter->lambda[i] = 0.0;
    filter->target_hist[i] = 0.0;
    filter->hist[i] = 0.0;
  }
  filter->patch = (MAT)NULL;
  filter->patch_hist = (MAT)NULL;
  filter->map.nrow = 0;
  filter->model_lambdas.nrow = 0;
  filter->flag = 1;
}


/*****************************************************
    Create a kernel of size (2*height+1)x(2*width+1)
    then put the pointers at the center
******************************************************/
MAT  Create_Kernel(int height, int  width)
{
  MAT  p;
  int  i;
  
  p=(float **)malloc((2*height+1)*sizeof(float *));
  p+=height;
  for(i=-height; i<=height; i++) {
    p[i]=(float *)malloc((2*width+1)*sizeof(float));
    p[i]+=width;
  }
  
  return(p);
}

void Free_Filter(FILTER *filter, FILTERBANK *ffamily)
{
  if (filter->id >=0) {
    free(filter->kernel);
    free(filter->rlist);
    free(filter->clist);
    free(filter->mark);
    free(filter->hist);
    free(filter->lambda);
    free(filter->target_hist);
    if (filter->patch != (MAT)NULL) {
      Free_Matrix(filter->patch, ffamily->greyLevel);
    }

    if (filter->patch_hist != (MAT)NULL) {
      Free_Matrix(filter->patch_hist, ffamily->greyLevel);
      filter->patch_hist = (MAT)NULL;
    }
    
    Free_Image_Float(&(filter->map));
    Free_Image_Float(&(filter->model_lambdas));
    filter->id = -1;
  }

}

/******************************************************/
void Normalize_Kernel(FILTER *filter)
{
  float sum, amp;
  int i,j;
  
  sum=0; amp=0;
  for(i=0; i<filter->size; i++)
    sum+=filter->kernel[i];
  sum /= ((float)filter->size);
  for(i=0; i<filter->size; i++)
    amp+=fabs(filter->kernel[i]-sum);
  
  for(i=0; i<filter->size; i++)
    filter->kernel[i]=(filter->kernel[i]-sum)*2.0/amp;
  
  return;
}

int Read_Filters(char *name, FILTERBANK *ffamily)
{
  int i,j, k, layer_no;
  FILE *IN;
  float ratio;
  float csize;
  
  if ( (IN=fopen(name,"rb"))==NULL) {
    printf("read filter error %s\n", name);
    return -1;
  }
  if (ffamily->nfilter > 0) {
    Free_Filter_Bank(ffamily);
  }
  
  fscanf(IN, "%d", &(ffamily->nfilter));
  printf("Loading %d filters from file '%s' ... \n",
	 ffamily->nfilter, name);
  ffamily->fbank = (FILTER *)malloc(sizeof(FILTER)*ffamily->nfilter);
  ffamily->filterDim = (int *)malloc(sizeof(int)*ffamily->nfilter);
  ffamily->longestBin =0;
  ffamily->totalBin = 0;
  ffamily->max_wid = 0;
  ffamily->max_ht = 0;
  ffamily->usedfilters = ffamily->nfilter;
  for(i=0; i< ffamily->nfilter; i++) {
    fscanf(IN, "%d", &layer_no);
    fscanf(IN, "%s", ffamily->fbank[i].name);
    fscanf(IN, "%d", &(ffamily->fbank[i].bin_num));
    /*  Kernel_List */
    ffamily->fbank[i].id = i+1;
    Read_Kernel(&(ffamily->fbank[i]), ffamily->fbank[i].name, 
		layer_no);
    ffamily->totalBin += ffamily->fbank[i].bin_num;
    if (i==0) {
      ffamily->filterDim[i] = 0;
    }
    else {
      ffamily->filterDim[i] = ffamily->filterDim[i-1] +
	ffamily->fbank[i-1].bin_num;
    }
    if (ffamily->longestBin < ffamily->fbank[i].bin_num)
      ffamily->longestBin = ffamily->fbank[i].bin_num;
    if (ffamily->max_wid < ffamily->fbank[i].wid)
      ffamily->max_wid = ffamily->fbank[i].wid;
    if (ffamily->max_ht < ffamily->fbank[i].ht)
      ffamily->max_ht = ffamily->fbank[i].ht;
  }
  printf("%d filters loaded successfully with total %d bins.\n", 
	 ffamily->nfilter, ffamily->totalBin);
  printf("Loading ");
  /* Read the window levels */
  
  ffamily->nlevel = 10;
  ratio = 1.5;
  if ( fscanf(IN, "%d", &(ffamily->nlevel)) ==1) {
    if (fscanf(IN,"%f", &ratio) == 0)
      ratio = 1.5;
    printf("user specfied ");
  }
  else {
    ffamily->nlevel = 10;
    printf("DEFAULT ");
  }
  
  ffamily->winLevels = (POINT2D *) malloc(sizeof(POINT2D)*ffamily->nlevel);
  j=0;
  ffamily->winLevels[0].x = ffamily->winLevels[0].y = 0;
  j++;
  csize = ratio * (2*ffamily->winLevels[0].x+1)*(2*ffamily->winLevels[0].y+1);
  for (i=1; i<ffamily->nlevel; i++) {
    do {
      j=j+1;
      k = (2*j+1)*(2*j+1);
    } while ((float)k < csize);
    if ( ((float)k/csize-1) > (1.0-(float)((2*j-1)*(2*j-1))/csize)) 
      j = j-1;
    ffamily->winLevels[i].x = ffamily->winLevels[i].y = j;
    csize = ratio * (2*ffamily->winLevels[i].x+1)*
      (2*ffamily->winLevels[i].y+1);
    j++;
  }
  
  printf("structure .... %d levels loaded\n", ffamily->nlevel);
  printf("Levels:\t");
  for (i=0; i<ffamily->nlevel; i++) {
    printf("%2d(%3d,%3d) ", i+1,
	   2*ffamily->winLevels[i].x+1, 2*ffamily->winLevels[i].y+1);
    if ( (i+1)%5 ==0 && i!= (ffamily->nlevel-1)) printf("\n\t");
  }
  printf("\n");
  ffamily->clevel = 0;
  
  fclose(IN);
  return ffamily->nfilter;
}

int Read_Lambdas(char *name, FILTERBANK *ffamily)
{
  int i,j,k;
  FILE *IN;
  float tmp;
  if ( (IN=fopen(name,"rb"))==NULL) {
    fprintf(stderr, "Cannot open lambda data file '%s'.\n", name);
    return -1;
  }
  k = 0;
  for(i=0; i< ffamily->nfilter; i++) {
    for (j=0; j < ffamily->fbank[i].bin_num; j++) {
      if (fscanf(IN,"%f", &(ffamily->fbank[i].lambda[j])) != 1) {
	fprintf(stderr,"Error when reading lambda file '%s' ", name);
	fprintf(stderr,"of filter %d at bin %d\n", i, j);
	fprintf(stderr,"Lambdas loaded are not complete.\n");
	fclose(IN);
	return -1;
      }
      k++;
    }
  }
  
  if (fscanf(IN,"%f", &tmp) == 1) {
    fprintf(stderr,"Warning: there are extra data in lambda file '%s'.\n",
	    name);
  }
  
  fclose(IN);
  return k;
}


void Scale_Lambdas(FILTERBANK *ffamily, float divid_factor)
{
  int i,j;
  for(i=0; i< ffamily->nfilter; i++) {
    for (j=0; j < ffamily->fbank[i].bin_num; j++) {
      ffamily->fbank[i].lambda[j] /= divid_factor;
    }
  }
  return;
}


int Read_Lambdas_Into_Vector(char *name, FILTERBANK *ffamily, 
			     ARRAY lambda_vec)
{
  int i,j,k;
  FILE *IN;
  float tmp;
  if ( (IN=fopen(name,"rb"))==NULL) {
    fprintf(stderr, "Cannot open lambda data file '%s'.\n", name);
    return -1;
  }
  k = 0;
  for(i=0; i< ffamily->nfilter; i++) {
    for (j=0; j < ffamily->fbank[i].bin_num; j++) {
      if (fscanf(IN,"%f", &(lambda_vec[ffamily->filterDim[i]+j])) != 1) {
	fprintf(stderr,"Error when reading lambda file '%s' ", name);
	fprintf(stderr,"of filter %d at bin %d\n", i, j);
	fprintf(stderr,"Lambdas loaded are not complete.\n");
	fclose(IN);
	return -1;
      }
      k++;
    }
  }
  
  if (fscanf(IN,"%f", &tmp) == 1) {
    fprintf(stderr,"Warning: there are extra data in lambda file '%s'.\n",
	    name);
  }
  
  fclose(IN);
  return k;
}

int Read_Multi_Lambdas_Into_Image(char *name, FILTERBANK *ffamily, 
				  IMAGEFLOAT *lambda_img,
				  int  num_lambda)
{
  int i,j,k, k1;
  FILE *IN;
  float tmp;
  if ( (IN=fopen(name,"rb"))==NULL) {
    fprintf(stderr, "Cannot open lambda data file '%s'.\n", name);
    return -1;
  }
  k = 0;
  for (k1=0; k1 < num_lambda; k1++) {
    for(i=0; i< ffamily->nfilter; i++) {
      for (j=0; j < ffamily->fbank[i].bin_num; j++) {
	if (fscanf(IN,"%f", 
		   &(lambda_img->data[k1][ffamily->filterDim[i]+j])) != 1) {
	  fprintf(stderr,"Error when reading lambda file '%s' ", name);
	  fprintf(stderr,"of filter %d at bin %d of %d\n", i, j, k1);
	  fprintf(stderr,"Lambdas loaded are not complete.\n");
	  fclose(IN);
	  return -1;
	}
	k++;
      }
    }
  }
  
  fclose(IN);
  return k;
}


int Save_Lambdas_Into_Vector(FILTERBANK *ffamily, 
			     ARRAY lambda_vec)
{
  int i,j,k;
  
  k = 0;
  for(i=0; i< ffamily->nfilter; i++) {
    for (j=0; j < ffamily->fbank[i].bin_num; j++) {
      lambda_vec[ffamily->filterDim[i]+j] = 
	ffamily->fbank[i].lambda[j];
      k++;
    }
  }
  return k;
}

void Free_Filter_Bank(FILTERBANK *ffamily)
{
  if (ffamily->nfilter > 0) {
    int i;
    for (i=0; i < ffamily->nfilter; i++) {
      Free_Filter(&(ffamily->fbank[i]), ffamily);
    }
    free(ffamily->fbank);
    free(ffamily->winLevels);
    free(ffamily->filterDim);
    ffamily->nfilter = 0;
  }
  return;
}


int Read_Filter_Marks(char *name, FILTERBANK *ffamily)
{
  int i, j, k;
  FILE *IN;
  
  if ( (IN=fopen(name,"rb")) == NULL) return -1;
  fscanf(IN, "%d", &i);
  if (i != ffamily->nfilter) {
    printf("Number of filters (%d) in filter mark file ", i);
    printf("does not match the current number of filters (%d).\n",
	   ffamily->nfilter);
    fclose(IN);
    return -1;
  }
  for (i=0; i<ffamily->nfilter; i++) {
    fscanf(IN, "%d%f", &j, &(ffamily->fbank[i].unit));
    if (i != j) {
      printf("Wrong bin mark file format. ");
      printf("Filter %d starts with %d instead.\n", i, j);
      fclose(IN);
      return -1;
    }
    for (k=0; k < ffamily->fbank[i].bin_num; k++) 
      fscanf(IN, "%f", &(ffamily->fbank[i].mark[k]));
  }
  
  /* Read out the histogram */
  for (i=0; i<ffamily->nfilter; i++) {
    fscanf(IN, "%d", &j);
    if (i != j) {
      printf("Wrong bin mark file format. ");
      printf("Filter %d starts with %d instead.\n", i, j);
      fclose(IN);
      return -1;
    }
    for (k=0; k < ffamily->fbank[i].bin_num; k++) 
      fscanf(IN, "%f", &(ffamily->fbank[i].hist[k]));
  }
  fclose(IN);
  printf("Histogram unit and bin marks are loaded successfully from '%s'.\n",
	 name);
  return 0;
}

int Save_Filter_Marks(char *name, FILTERBANK *ffamily)
{     
  int i,k;
  FILE *OUT;

  if ( (OUT=fopen(name,"wb")) == NULL) return -1;
  fprintf(OUT, "%d\n", ffamily->nfilter);
  for (i=0; i<ffamily->nfilter; i++) {
    fprintf(OUT, "%d %f ",i, ffamily->fbank[i].unit);
    for (k=0; k < ffamily->fbank[i].bin_num; k++) 
      fprintf(OUT, "%f ", ffamily->fbank[i].mark[k]);
    fprintf(OUT,"\n");
  }
  /* Save the histogram also */
  fprintf(OUT,"\n\n");
  for (i=0; i<ffamily->nfilter; i++) {
    fprintf(OUT, "%d ",i);
    for (k=0; k < ffamily->fbank[i].bin_num; k++) {
      fprintf(OUT, "%f ", ffamily->fbank[i].hist[k]);
    }
    fprintf(OUT,"\n");
  }

  fclose(OUT);
  printf("Histogram unit and bin marks are saved successfully into '%s'\n",
	 name);
  return 0;
}

int Generate_Marks_From_Image(FILTERBANK *ffamily, 
			      IMAGEINT *animage,
			      CONTROL_INFO *controlInfo)
{
  MAT  Resp;
  int ht, wid;
  ARRAY hist;
  float minTail;
  int i, i1, i2, k, k1;
  int r, c;
  FILTER *filter;
  float min, max, sum, ftmp;
  int adjustTail;
  int index;
  int trials;
  hist = Create_Array(ffamily->longestBin);
  Resp = Create_Matrix(animage->nrow, animage->ncol);
  
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    ht = filter->ht; wid=filter->wid; 
    ht = 0; wid = 0;
    minTail = 0.1 * (float)((animage->nrow-2*ht)*(animage->ncol-2*wid))/ 
      (float)filter->bin_num;
    min=1000; max=-10000;
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	sum=0.0;    
	for(i=0; i<filter->size; i++) { 
	  i1 = (r+filter->rlist[i]+animage->nrow)%animage->nrow;
	  i2 = (c+ filter->clist[i]+animage->ncol)%animage->ncol;
	  sum += filter->kernel[i]*
	    animage->data[i1][i2]; 
	}
	if(min>sum) min=sum;
	if(max<sum) max=sum;
	
	Resp[r][c]=sum;
      }
    }
    if( (filter->wid==0)&&(filter->ht==0)) { /* the intensity histogram */
      for(i=0; i<filter->bin_num; i++)
	filter->mark[i]= i*(float)(max-min)/(float)filter->bin_num+min;
      filter->unit=(float)(max-min+1)/(float)filter->bin_num;
    }
    else { /* Not an intensity filter */
     
      trials = 0;
      if ((max-min) < 0.0001) {
	max += 0.6;
	min -= 0.5;
	for(i=0; i<filter->bin_num; i++)
	  filter->mark[i]=((float)i)*(max-min)/((float)filter->bin_num) +
	    min;
	filter->unit = (max-min)/((float)filter->bin_num);
      }
      else {
	do {
	  adjustTail = 0;
	  for(i=0; i<filter->bin_num; i++)
	    filter->mark[i]=((float)i)*(max-min)/((float)filter->bin_num) +
	      min;
	  filter->unit = (max-min)/((float)filter->bin_num);

	  for(i=0; i<filter->bin_num; i++) hist[i]=0.0;
	  for(r=ht; r<animage->nrow-ht; r++) {
	    for(c=wid; c<animage->ncol-wid; c++) {
	      index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	      if(index<0) {
		hist[0]+=1; continue;
	      }
	      if(index>=filter->bin_num-1) {
		hist[filter->bin_num-1]+=1; continue;
	      }
	      hist[index]+=1;
	    }
	  }
	  
	  if (hist[0] < minTail || hist[filter->bin_num-1] < minTail) {
	    /*printf("Hist needs to be adjusted: ");
	      printf("min = %6.4f max=%6.4f unit=%6.4f\n",
	      min, max, filter->unit);
	      printf("Histogram: ");
	      for (i=0; i<bin_num; i++) {
	      printf("%5.1f ", hist[i]);
	      }
	      printf("\n");*/
	    adjustTail = 1;
	  }
	  if (hist[0] < minTail) {
	    for (i=1; i < filter->bin_num; i++) {
	      if ( (hist[0]+hist[i]) > minTail) {
		min += 0.05*(minTail-hist[0])*filter->unit/hist[i];
		min += 0.0005*filter->unit;
		break;
	      }
	      min += filter->unit;
	      hist[0] += hist[i];
	    }
	  }
	  if (hist[filter->bin_num-1] < minTail) {
	    for (i=filter->bin_num-2; i >= 1; i--) {
	      if ( (hist[filter->bin_num-1]+hist[i]) > minTail) {
		max -= 0.05*(minTail-hist[filter->bin_num-1])*
		  filter->unit/hist[i];
		max -= 0.0005*filter->unit;
		break;
	      }
	      max -= filter->unit;
	      hist[filter->bin_num-1] += hist[i];
	    }
	  }
	  trials++;
	  if (trials >=100) break;
	} while(adjustTail);
      }
    }
    for(i=0; i<filter->bin_num; i++) hist[i]=0.0;  /* clean up */
    
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index > filter->bin_num-1) {
	    index = filter->bin_num-1;
	  }
	}
	hist[index]+=1;
      }
    }
    
    printf("\tParameters: ");
    printf("min = %6.4f max=%6.4f unit=%6.4f\n", min, max, filter->unit);
    
    if( (filter->wid==0)&&(filter->ht==0)) { /* the intensity histogram */
      if (filter->sum_min > min)
	filter->sum_min = min;
      if (filter->sum_max < max)
	filter->sum_max = max;
    }
    else {
      filter->sum_min += min;
      filter->sum_max += max;
    }
    /* Also save the histogram */
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->hist[i] = hist[i]/sum;
    }
    for (i=0; i< filter->bin_num; i++) {
      printf("%6.4f ", filter->hist[i]);
    }
    printf("\n");
  }
  Free_Array(hist);
  Free_Matrix(Resp, animage->nrow);
  return 0;
}

int Generate_Target_Hist_From_Image(FILTERBANK *ffamily, 
				    IMAGEINT *animage)
{
  MAT  Resp;
  int ht, wid;
  int i, i1, i2, k;
  int r, c;
  FILTER *filter;
  float min, max, sum;
  int index;
  int total_pixels=0;
  Resp = Create_Matrix(animage->nrow, animage->ncol);
  
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    ht = filter->ht; wid=filter->wid;
    /* Force to use circular boundary in the observed */
    ht = 0;
    wid = 0;
   
    total_pixels=0;
    /* Calculate the filter response first */
     min=1000; max=-10000; 
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	sum=0.0;    
	for(i=0; i<filter->size; i++) { 
	  i1 = (r+filter->rlist[i]+animage->nrow*100)%animage->nrow;
	  i2 = (c+ filter->clist[i]+animage->ncol*100)%animage->ncol;
	  sum += filter->kernel[i]*
	    animage->data[i1][i2]; 
	}
	
	if(min>sum) min=sum;
	if(max<sum) max=sum;  
	Resp[r][c]=sum;
	total_pixels++;
      }
    }
    printf("\tFilter response: min=%f max=%f unit = %f mark[0] = %f ",
         min, max, filter->unit, filter->mark[0]);  
    printf("%d valid pixels.\n", total_pixels);
    for(i=0; i<filter->bin_num; i++) 
      filter->target_hist[i]=0.0;  /* clean up */
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index >= filter->bin_num) {
	    index = filter->bin_num-1;
	  }
	}
	filter->target_hist[index]+=1;
      }
    }
    /* Also save the histogram */
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += filter->target_hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->target_hist[i] /= sum;
    }
    printf("\tFinal Histogram: ");
    for (i=0; i< filter->bin_num; i++) {
      printf("%6.4f ", filter->target_hist[i]);  
    }
    printf("\n");
  }
  Free_Matrix(Resp, animage->nrow);
  return 0;
}

int Generate_Target_Hist_From_Image_Analytical(FILTERBANK *ffamily, 
					       IMAGEINT *animage,
					       CONTROL_INFO *controlInfo)
{
  MAT  Resp;
  int ht, wid;
  int i, i1, i2, k;
  int r, c;
  FILTER *filter;
  float *hist;
  float *hist_tmp;
  int bin;
  float min, max, sum;
  int index;
  double  sum_power_4, sum_power_3, sum_power_2, sum_power_1;
  double  double_sum, sum_tmp;
  int   total_pixels=0;
  int invalid;
  float p_val, c_val;
  float   tmp1, tmp2, tmp3;
  int analytical=0;
  float total_dist;
  float chi_dist;
  float *chi_dist_arr;
  float L1_dist, *L1_dist_arr;

  total_dist = 0;
  chi_dist_arr = (float *)malloc(sizeof(float)*ffamily->nfilter);
  L1_dist_arr = (float *)malloc(sizeof(float)*ffamily->nfilter);
  Resp = Create_Matrix(animage->nrow, animage->ncol);
  
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    hist = filter->target_hist;
    hist_tmp = (float *)malloc(sizeof(float)*filter->bin_num);
    ht = filter->ht; wid=filter->wid;
    ht =0;
    wid =0;
    /* Calculate the filter response first */
    min=1000; max=-10000; 
    total_pixels=0;
    sum_power_4 =0.0;
    sum_power_3 =0.0;
    sum_power_2 =0.0;
    sum_power_1 =0.0;
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	sum=0.0;    
	for(i=0; i<filter->size; i++) { 
	  i1 = (r+filter->rlist[i]+animage->nrow)%animage->nrow;
	  i2 = (c+ filter->clist[i]+animage->ncol)%animage->ncol;
	  sum += filter->kernel[i]*
	    animage->data[i1][i2]; 
	}
	if(min>sum) min=sum;
	if(max<sum) max=sum;  
	Resp[r][c]=sum;
       }
     }
    printf("\n\tFilter response: min=%f max=%f unit = %f mark[0] = %f\n",
	   min, max, filter->unit, filter->mark[0]);  
    for(i=0; i<filter->bin_num; i++) 
      filter->target_hist[i]=0.0;  /* clean up */
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	total_pixels++;
	index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index >= filter->bin_num) {
	    index = filter->bin_num-1;
	  }
	}
	filter->target_hist[index]+=1;
	double_sum = (double)Resp[r][c];
	if (isnan(double_sum) || isinf(double_sum) || finite(double_sum)==0) {
	  printf("Invalid double_sum %15.13lf detected and", double_sum);
	  printf("was changed to zero forcesfully.\n");
	  /*printf("Press a digit to continue .... ");
	    scanf("%d", &bin);
	    fflush(stdin);
	  */
	  double_sum = 0.0;
	}
	sum_power_1 += double_sum;
	sum_tmp = double_sum * double_sum;
	sum_power_2 += sum_tmp;
	sum_power_3 += sum * sum_tmp;
	sum_power_4 += sum_tmp * sum_tmp;
      }
    }
    /* Also save the histogram */
    /* Calculate the p and c values */
    sum_power_1 /= (double)total_pixels; 
    sum_power_2 /= (double)total_pixels; 
    sum_power_3 /= (double)total_pixels; 
    sum_power_4 /= (double)total_pixels; 
    sum_tmp = sum_power_1 * sum_power_1;
    double_sum = sum_power_4 - 4.0 * sum_power_1 * sum_power_3 +
      6.0 * sum_tmp * sum_power_2 -
      3.0 * sum_tmp * sum_tmp;
    sum_tmp = sum_power_2 - sum_tmp; 
    /* Correct the bias */
    double_sum /= (sum_tmp*sum_tmp);
    sum_tmp *= (double)(total_pixels)/(double)(total_pixels-1);  
    
    printf("Kurtosis %15.13lf Var: %15.13lf with total %d \n", 
	   double_sum, sum_tmp, total_pixels);
    invalid  = 0;
    if (isnan(double_sum) || isinf(double_sum) || finite(double_sum)==0) {
      printf("Invalid kurtosis %15.13lf Var: %15.13lf with total %d \n", 
	     double_sum, sum_tmp, total_pixels);
      printf("Sum_1 = %15.13lf Sum_2 = %15.13lf ",
	     sum_power_1,sum_power_2);
      printf("Sum_3 = %15.13lf Sum_4 = %15.13lf\n",
	     sum_power_3, sum_power_4);
      invalid = 1;
    }
    if( (filter->wid==0)&&(filter->ht==0)) {
      invalid = 1;
    }
    if (invalid ==0) {
      p_val = 3.0/(double_sum-3.0);
      c_val = sum_tmp / p_val  ;
      
      printf("Filter %d and bin= %d p_val = %12.10f c_val=%12.10f ", 
	     k,  filter->bin_num,
	     p_val, c_val);
      if ( (p_val < 0.0) || (c_val < 0.0)) {
	invalid = 1;
      }
    }
    if (invalid ==0) {
      sum_power_1 = 0.0;
      tmp1 = p_val - 0.5;
      tmp3 = sqrt(2/c_val);
      for (bin=0; bin  < filter->bin_num; bin++) {
	tmp2 = fabs(filter->mark[bin]);
	double_sum = pow(tmp2, tmp1);
	tmp2 *= tmp3;
	double_sum *= bessik(tmp1, tmp2);
	if (isnan(double_sum) || isinf(double_sum) || finite(double_sum)==0) {
	  invalid = 1;
	  break;
	}
	hist_tmp[bin] = (float)double_sum;
	sum_power_1 += double_sum;
	/*printf("Bin %2d => %12.10f %12.10f\n", bin, filter->mark[bin],
	  hist[bin]);*/
      }
      if (isnan(sum_power_1) || isinf(sum_power_1) || 
	  finite(sum_power_1)==0) {
	invalid = 1;
      }
    }
    
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += filter->target_hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->target_hist[i] /= sum;
    }
    printf("\nFinal Histogram (Computed Histogram): \n\t");
    for (i=0; i< filter->bin_num; i++) {
      printf("%6.4f ", filter->target_hist[i]);  
    }
    printf("\n");
    if ((invalid == 0) && ((filter->ht+filter->wid)>0 ) ) {
      
      for (bin=0; bin  < filter->bin_num; bin++) {
	hist_tmp[bin] = hist_tmp[bin]/(float)sum_power_1;
      }
      chi_dist = Calc_Hist_Divergence(filter->target_hist,
				      hist_tmp,
				      filter->bin_num);
      chi_dist_arr[k] = chi_dist;
      L1_dist = Calc_Hist_Dist_L1(filter->target_hist,
				  hist_tmp,
				  filter->bin_num);
      L1_dist_arr[k] = L1_dist;
      if (L1_dist <= controlInfo->accept_chi_threshold) {
	total_dist += chi_dist;
      }
      else {
	filter->flag = 0;
      }
      analytical += filter->flag;
      if (filter->flag==0) {
	printf("Filter is NOT used. ");
	printf("Final Histogram (Analytical form ");
	printf("with %8.6f L1-dist and %8.6f chi-dist): \n\t",
	       L1_dist, chi_dist);
      }
      else {
	printf("Final Histogram (Analytical form ");
	printf("is used %8.6f L1-dist and %8.6f chi-dist): \n\t",
	       L1_dist, chi_dist);
      }
      if (controlInfo->use_analytical_form ==3) {
        printf("Computed histogram will be used instead.\n");
      } 
      for (i=0; i< filter->bin_num; i++) {
        if (controlInfo->use_analytical_form !=3)
	   filter->target_hist[i] = hist_tmp[i];
	printf("%6.4f ", filter->target_hist[i]);  
      }
      printf("\n");
    }
    else {
      L1_dist_arr[k] = 0.0;
      chi_dist_arr[k] = 0.0;
    }
    free(hist_tmp);
  }
  Free_Matrix(Resp, animage->nrow);
  printf("%d/%d filters used analytical forms for its histograms.\n",
	 analytical, ffamily->nfilter);
  if (analytical > 0) {
    printf("For %d analytical filters, the total L1-distance is %8.6f ",
	  analytical, total_dist);
    printf(" and average is %8.6f\n",
	   (float)(total_dist/analytical));
  
    for (k=0; k < ffamily->nfilter; k++) {
      printf("%8.6f(%8.6f %d) ", L1_dist_arr[k], chi_dist_arr[k],
	     ffamily->fbank[k].flag);
    }
    printf("\n");
  }
  ffamily->usedfilters = 0;
  for (k=0; k < ffamily->nfilter; k++) {
    ffamily->usedfilters += (ffamily->fbank[k].flag == 1);
  }
  printf("Total used filters: %d\n", ffamily->usedfilters);
  free((void *)chi_dist_arr);
  free((void *)L1_dist_arr);
  return 0;
}

int Generate_Target_Hist_From_Image_Analytical_All(FILTERBANK *ffamily, 
						   IMAGEINT *animage,
						   CONTROL_INFO *controlInfo)
{
  MAT  Resp;
  int ht, wid;
  int i, i1, i2, k;
  int r, c;
  FILTER *filter;
  float *hist;
  float *hist_tmp;
  int bin;
  float min, max, sum;
  int index;
  double  sum_power_4, sum_power_3, sum_power_2, sum_power_1;
  double  double_sum, sum_tmp;
  int   total_pixels=0;
  int invalid;
  float p_val, c_val;
  float   tmp1, tmp2, tmp3;
  int analytical=0;
  float total_dist;
  float chi_dist;
  float *chi_dist_arr;
  float L1_dist, *L1_dist_arr;
  chi_dist_arr = (float *)malloc(sizeof(float)*ffamily->nfilter);
  L1_dist_arr = (float *)malloc(sizeof(float)*ffamily->nfilter);
  total_dist = 0;
  
  Resp = Create_Matrix(animage->nrow, animage->ncol);
  
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    hist = filter->target_hist;
    hist_tmp = (float *)malloc(sizeof(float)*filter->bin_num);
    ht = filter->ht; wid=filter->wid;
    ht =0; wid=0;
    /* Calculate the filter response first */
    min=1000; max=-10000; 
    total_pixels=0;
    sum_power_4 =0.0;
    sum_power_3 =0.0;
    sum_power_2 =0.0;
    sum_power_1 =0.0;
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	sum=0.0;    
	for(i=0; i<filter->size; i++) { 
	  i1 = (r+filter->rlist[i]+animage->nrow)%animage->nrow;
	  i2 = (c+ filter->clist[i]+animage->ncol)%animage->ncol;
	  sum += filter->kernel[i]*
	    animage->data[i1][i2]; 
	}
	if(min>sum) min=sum;
	if(max<sum) max=sum;  
	Resp[r][c]=sum;
       }
     }
    printf("\n\tFilter response: min=%f max=%f unit = %f mark[0] = %f\n",
	   min, max, filter->unit, filter->mark[0]);  
    for(i=0; i<filter->bin_num; i++) 
      filter->target_hist[i]=0.0;  /* clean up */
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	total_pixels++;
	index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index >= filter->bin_num) {
	    index = filter->bin_num-1;
	  }
	}
	filter->target_hist[index]+=1;
	double_sum = (double)Resp[r][c];
	if (isnan(double_sum) || isinf(double_sum) || finite(double_sum)==0) {
	  printf("Invalid double_sum %15.13lf detected and", double_sum);
	  printf("was changed to zero forcesfully.\n");
	  /*printf("Press a digit to continue .... ");
	    scanf("%d", &bin);
	    fflush(stdin);
	  */
	  double_sum = 0.0;
	}
	/* sum_power_1 is the sum of x.
	   sum_power_2 is the sum of x^2.
	   sum_power_3 is the sum of x^3.
	   sum_power_4 is the sume of x^4.
	*/
	sum_power_1 += double_sum;
	sum_tmp = double_sum * double_sum;
	sum_power_2 += sum_tmp;
	sum_power_3 += sum * sum_tmp;
	sum_power_4 += sum_tmp * sum_tmp;
      }
    }
    /* Also save the histogram */
    /* Calculate the p and c values */
    /* Calculate the average of the histograms */
    sum_power_1 /= (double)total_pixels; 
    sum_power_2 /= (double)total_pixels; 
    sum_power_3 /= (double)total_pixels; 
    sum_power_4 /= (double)total_pixels; 
    sum_tmp = sum_power_1 * sum_power_1;
    /* Kurtosis = E((x-u)^4)/(E((x-u)^2)^2)
       E((x-u)^4) = E(x^4) - 4 u E(x^3) + 6 u^2 E(x^2) - 3 u^4 */
    double_sum = sum_power_4 - 4.0 * sum_power_1 * sum_power_3 +
      6.0 * sum_tmp * sum_power_2 -
      3.0 * sum_tmp * sum_tmp;
    sum_tmp = sum_power_2 - sum_tmp; 
    /* Correct the bias */
    /* This equation matches the result from the matlab */
    /* double_sum is the kortosis */
    double_sum /= (sum_tmp*sum_tmp);
    /* Normalized variance */
    sum_tmp *= (double)(total_pixels)/(double)(total_pixels-1);  
    if (double_sum < 3.05)
      double_sum = 3.05;
    printf("Kurtosis %15.13lf Var: %15.13lf with total %d \n", 
	   double_sum, sum_tmp, total_pixels);
    invalid  = 0;
    if( (filter->wid==0)&&(filter->ht==0)) {
      invalid = 1;
    }
    if ((invalid == 0) &&
	(isnan(double_sum) || isinf(double_sum) || finite(double_sum)==0)) {
      printf("Invalid kurtosis %15.13lf Var: %15.13lf with total %d \n", 
	     double_sum, sum_tmp, total_pixels);
      printf("Sum_1 = %15.13lf Sum_2 = %15.13lf ",
	     sum_power_1,sum_power_2);
      printf("Sum_3 = %15.13lf Sum_4 = %15.13lf\n",
	     sum_power_3, sum_power_4);
      invalid = 1;
    }
   
    if (invalid ==0) {
      p_val = 3.0/(double_sum-3.0);
      c_val = sum_tmp / p_val  ;
      
      printf("Filter %d and bin= %d p_val = %12.10f c_val=%12.10f ", 
	     k,  filter->bin_num,
	     p_val, c_val);
      if ( (p_val < 0.0) || (c_val < 0.0)) {
	invalid = 1;
      }
    }
    if (invalid ==0) {
      sum_power_1 = 0.0;
      tmp1 = p_val - 0.5;
      tmp3 = sqrt(2/c_val);
      for (bin=0; bin  < filter->bin_num; bin++) {
	tmp2 = fabs(filter->mark[bin]);
	sum_power_2 = pow(tmp2, tmp1);
	tmp2 *= tmp3;
	sum_power_2 *= bessik(tmp1, tmp2);
	if (isnan(sum_power_2) || isinf(sum_power_2) 
	    || finite(sum_power_2)==0) {
	  invalid = 1;
	  printf("Sampling is not valid with sum = %lf at bin %d\n", 
		 sum_power_2, bin);
	  break;
	}
	hist_tmp[bin] = (float)sum_power_2;
	sum_power_1 += sum_power_2;
	/*printf("Bin %2d => %12.10f %12.10f\n", bin, filter->mark[bin],
	  hist[bin]);*/
      }
      if (isnan(sum_power_1) || isinf(sum_power_1) || 
	  finite(sum_power_1)==0) {
	invalid = 1;
	printf("Sampling is not valid with sum = %lf\n", sum_power_1);
      }
      if (p_val > 3.0) 
	sum_power_3 = 0.005;
      else
	sum_power_3 = 0-0.01;
      while (invalid != 0) {
	invalid =0;
	sum_power_1 = 0.0;
	
	double_sum += sum_power_3;
	
	p_val = 3.0/(double_sum-3.0);
	c_val = sum_tmp / p_val  ;
	
	tmp1 = p_val - 0.5;
	tmp3 = sqrt(2/c_val);
	for (bin=0; bin  < filter->bin_num; bin++) {
	  tmp2 = fabs(filter->mark[bin]);
	  sum_power_2 = pow(tmp2, tmp1);
	  tmp2 *= tmp3;
	  sum_power_2 *= bessik(tmp1, tmp2);
	  if (isnan(sum_power_2) || isinf(sum_power_2) 
	      || finite(sum_power_2)==0) {
	    invalid = 1;
	    printf("Sampling is not valid with sum = %lf at bin %d\n", 
		   sum_power_2, bin);
	    break;
	  }
	  hist_tmp[bin] = (float)sum_power_2;
	  sum_power_1 += sum_power_2;
	  /*printf("Bin %2d => %12.10f %12.10f\n", bin, filter->mark[bin],
	    hist[bin]);*/
	  
	}
	printf("Filter %d and bin= %d p_val = %12.10f c_val=%12.10f ", 
	       k,  filter->bin_num,
	       p_val, c_val);
      }
      
    }
    
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += filter->target_hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->target_hist[i] /= sum;
    }
    printf("\nFinal Histogram (Computed Histogram): \n\t");
    for (i=0; i< filter->bin_num; i++) {
      printf("%6.4f ", filter->target_hist[i]);  
    }
    printf("\n");
    if ((invalid == 0) && ((filter->ht+filter->wid)>0 ) ) {
     
      for (bin=0; bin  < filter->bin_num; bin++) {
	hist_tmp[bin] = hist_tmp[bin]/(float)sum_power_1;
      }
      chi_dist = Calc_Hist_Divergence(filter->target_hist,
				      hist_tmp,
				      filter->bin_num);
      chi_dist_arr[k] = chi_dist;
      L1_dist = Calc_Hist_Dist_L1(filter->target_hist,
				  hist_tmp,
				  filter->bin_num);
      L1_dist_arr[k] = L1_dist;
      if (L1_dist <= controlInfo->accept_chi_threshold) {
	total_dist += L1_dist;
      }
      else {
	filter->flag = 0;
      }
      analytical += filter->flag;
      if (filter->flag==0) {
	printf("Filter is NOT used. ");
	printf("Final Histogram (Analytical form ");
	printf("with %8.6f L1-dist and %8.6f chi-dist): \n\t",
	       L1_dist, chi_dist);
      }
      else {
	printf("Final Histogram (Analytical form ");
	printf("is used %8.6f L1-dist and %8.6f chi-dist): \n\t",
	       L1_dist, chi_dist);
      }
      if (controlInfo->use_analytical_form ==3) {
        printf("Computed histogram will be used instead.\n");
      }
 
      for (i=0; i< filter->bin_num; i++) {
        if (controlInfo->use_analytical_form != 3)
	  filter->target_hist[i] = hist_tmp[i];
	printf("%6.4f ", filter->target_hist[i]);  
      }
      printf("\n");
      if (controlInfo->tail_factor > 1.0) {
	printf("Emphasizing the tail with factor %6.4f\n",
	       controlInfo->tail_factor);
	p_val = Find_Power_For_Tail_Factor(filter->bin_num,
					   controlInfo->tail_factor,
					   controlInfo->tail_epsilon);
	sum = 0.0;
	
	for (i=0; i< filter->bin_num; i++) {
	  if (i == (((int)(filter->bin_num-1)/2))) {
	    filter->target_hist[i] *= 
	       (pow(fabs((double)(1/2.0)), p_val)+1.0);
	    printf("%8.6f ", pow(fabs((double)(1/2.0)), p_val)+1.0);
	  }
	  else {
	    filter->target_hist[i] *= 
	      (pow(fabs((double)(i-(double)(filter->bin_num-1.0)/2.0)),
		   p_val)+1.0);
	    printf("%8.6f ", pow(fabs((double)(i-(double)(filter->bin_num-1.0)/2.0)),
				p_val)+1.0);
	  }
	  
	  sum += filter->target_hist[i];
	}
	printf("\n");
	for (i=0; i< filter->bin_num; i++) {
	  filter->target_hist[i] /= sum;
	}
	printf("Tailed version: \n\t");
	for (i=0; i< filter->bin_num; i++) {
	  printf("%6.4f ", filter->target_hist[i]);  
	}
	printf("\n");
      }
    }
    else {
      L1_dist_arr[k] = 0.0;
      chi_dist_arr[k] = 0.0;
    }
    free(hist_tmp);
  }
  Free_Matrix(Resp, animage->nrow);
  printf("%d/%d filters used analytical forms for its histograms.\n",
	 analytical, ffamily->nfilter);
  if (analytical > 0) {
    printf("For %d analytical filters, the total L1-distance is %8.6f ",
	   analytical, total_dist);
    printf(" and average is %8.6f\n",
	   (float)(total_dist/analytical));
    for (k=0; k < ffamily->nfilter; k++) {
      printf("%8.6f(%8.6f %d) ", L1_dist_arr[k], chi_dist_arr[k],
	     ffamily->fbank[k].flag);
    }
    printf("\n");
  }
  ffamily->usedfilters = 0;
  for (k=0; k < ffamily->nfilter; k++) {
    ffamily->usedfilters += (ffamily->fbank[k].flag == 1);
  }
  printf("Total used filters: %d\n", ffamily->usedfilters);
  free((void *)chi_dist_arr);
  free((void *)L1_dist_arr);
  return 0;
}

void Initial_Synthesize_Image(FILTERBANK *ffamily, 
			      IMAGEINT *animage,
			      IMAGEINT *holeimage)
     
{
  int i,j;
  int val;
  
  for (i=0; i < animage->nrow; i++) {
    for (j=0; j < animage->ncol; j++) {
      if (holeimage->data[i][j] >0) {
	val = (int)(Unit_Random()*ffamily->greyLevel);
	if (val >= ffamily->greyLevel)
	  val = ffamily->greyLevel - 1;
	animage->data[i][j] = val;
      }
    }
  }
  return;
}


int Generate_Target_Hist_From_Image_WithHoles(FILTERBANK *ffamily, 
					      IMAGEINT *animage,
					      IMAGEINT *holeimage)
{
  MAT  Resp;
  int ht, wid;
  int i, i1, i2, k;
  int r, c;
  FILTER *filter;
  float min, max, sum;
  int index;
  
  Resp = Create_Matrix(animage->nrow, animage->ncol);
  
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    ht = filter->ht; wid=filter->wid;
    /* Calculate the filter response first */
     min=1000; max=-10000; 
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	sum=0.0;    
	for(i=0; i<filter->size; i++) { 
	  i1 = (r+filter->rlist[i]+animage->nrow)%animage->nrow;
	  i2 = (c+ filter->clist[i]+animage->ncol)%animage->ncol;
	  sum += filter->kernel[i]*
	    animage->data[i1][i2]; 
	}
      if(min>sum) min=sum;
      if(max<sum) max=sum;  
       Resp[r][c]=sum;

      }
    }
    printf("\tFilter response: min=%f max=%f unit = %f mark[0] = %f\n",
         min, max, filter->unit, filter->mark[0]);  
    for(i=0; i<filter->bin_num; i++) 
      filter->target_hist[i]=0.0;  /* clean up */
    for(r=ht; r<animage->nrow-ht; r++) {
      for(c=wid; c<animage->ncol-wid; c++) {
	if (holeimage->data[r][c]>0) {
	  index=Round((Resp[r][c]-filter->mark[0])/(filter->unit));
	  if(index<0) {
	    index = 0;
	  }
	  else {
	    if(index >= filter->bin_num) {
	      index = filter->bin_num-1;
	    }
	  }
	  
	  filter->target_hist[index] += holeimage->data[r][c];
	}
      }
    }
    /* Also save the histogram */
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += filter->target_hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->target_hist[i] /= sum;
    }
    printf("\tFinal Histogram: ");
    for (i=0; i< filter->bin_num; i++) {
      printf("%6.4f ", filter->target_hist[i]);  
    }
    printf("\n");
  }
  Free_Matrix(Resp, animage->nrow);
  return 0;
}


int Scale_Target_Hist(FILTERBANK *ffamily, float norm)
{
  int i;
  int k;
  FILTER *filter;
  float sum;
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    sum = 0.0;
    for (i=0; i< filter->bin_num; i++) {
      sum += filter->target_hist[i];
    }
    for (i=0; i< filter->bin_num; i++) {
      filter->target_hist[i] *= (norm/sum);
    }
  }
}


int Generate_Hist_From_Map(FILTERBANK *ffamily)
{
  
  
  int k,i;
  int r, c;
  FILTER *filter;
  int index;
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    if (filter->flag ==0) continue;
    for(i=0; i<filter->bin_num; i++) 
      filter->hist[i]=0.0;  /* clean up */
    for(r=0; r<filter->map.nrow; r++) {
      for(c=0; c<filter->map.ncol; c++) {
	index=Round((filter->map.data[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index >= filter->bin_num) {
	    index = filter->bin_num-1;
	  }
	}
	filter->hist[index]+=1;
      }
    }
  }
  return 0;
}


int Generate_Hist_From_Map_Hole(FILTERBANK *ffamily,
				ONE_HOLE *thepatch)
{
  
  
  int k,i;
  int r, c;
  FILTER *filter;
  int index;
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    for(i=0; i<filter->bin_num; i++) 
      filter->hist[i]=0.0;  /* clean up */
    for(r=thepatch->row; r< (thepatch->row+thepatch->hole_size); r++) {
      for(c=thepatch->col; c<(thepatch->col+thepatch->hole_size); c++) {
	index=Round((filter->map.data[r][c]-filter->mark[0])/(filter->unit));
	if(index<0) {
	  index = 0;
	}
	else {
	  if(index >= filter->bin_num) {
	    index = filter->bin_num-1;
	  }
	}
	filter->hist[index]+=1;
      }
    }
  }
  return 0;
}


void Initialize_Obs(FILTERBANK *ffamily, IMAGEINT *obs,
		    CONTROL_INFO *controlInfo,
		    IMAGE_DATABASE *adatabase)
{
  int i,j,k;
  FILTER *filter;
  if (Read_An_Obs_Image(controlInfo->inputFile,
		    obs,
		    controlInfo) !=0) {
    fprintf(stderr,"Could not load image '%s'.\n",
	    controlInfo->inputFile);
    exit(-1);
  }
  for (k=0; k < adatabase->nimages; k++) {
    if (strcmp(controlInfo->inputFile,
	       adatabase->entries[k].fname) ==0) {
      break;
    }
  }
  if (k < adatabase->nimages) {
    printf("Observed image is the %dth image in the database.\n",
	   k);
    printf("Loading target histogram from the database.\n");
    for (i=0; i < ffamily->nfilter; i++) {
      filter = &(ffamily->fbank[i]);
      for (j=0; j < filter->bin_num; j++) 
	filter->target_hist[j] = 
	  adatabase->entries[k].fvector[ffamily->filterDim[i]+j];
    }
  }
  else {
    if (controlInfo->use_analytical_form) {
      Generate_Target_Hist_From_Image_Analytical(ffamily, obs, controlInfo);
    } 
    else {
      Generate_Target_Hist_From_Image(ffamily, obs);
    }
  }
  return;
}

void Initialize_Obs_Analytical(FILTERBANK *ffamily, IMAGEINT *obs,
			       CONTROL_INFO *controlInfo)
{
  int i,j,k;
  FILTER *filter;
  if (Read_An_Obs_Image(controlInfo->inputFile,
		    obs,
		    controlInfo) !=0) {
    fprintf(stderr,"Could not load image '%s'.\n",
	    controlInfo->inputFile);
    exit(-1);
  }
  
  if (controlInfo->use_analytical_form) {
    if (controlInfo->use_analytical_form >= 2) {
      Generate_Target_Hist_From_Image_Analytical_All(ffamily, obs, 
						     controlInfo);
    }
    else {
      Generate_Target_Hist_From_Image_Analytical(ffamily, obs,
						 controlInfo);
    }
  } 
  else {
    Generate_Target_Hist_From_Image(ffamily, obs);
  }
  return;
}

   
void Initialize_Obs_Satellite(FILTERBANK *ffamily, IMAGEINT *obs,
			      CONTROL_INFO *controlInfo,
			      IMAGE_DATABASE *adatabase)
{
  int i,j,k;
  FILTER *filter;
  float *distDb, *distMax;
  float *distRandom;
  int   *orderList;
  float maxDist, dist;
  float *fvector;
  float *random_vector;
  int   start_index;
  IMAGEINT random_img;
  
  random_img.nrow = 0;
  if (Read_An_Obs_Image(controlInfo->inputFile,
		    obs,
		    controlInfo) !=0) {
    fprintf(stderr,"Could not load image '%s'.\n",
	    controlInfo->inputFile);
    exit(-1);
  }
  if (controlInfo->num_ref_model == 0) {
    Generate_Target_Hist_From_Image(ffamily, obs);
    return; 
  }
  for (k=0; k < adatabase->nimages; k++) {
    if (strcmp(controlInfo->inputFile,
	       adatabase->entries[k].fname) ==0) {
      break;
    }
  }
  random_img.data = Create_Noise_Matrix(obs->nrow,
					obs->ncol,
					(int)0,
					(controlInfo->greyLevel-1));
  random_img.nrow = obs->nrow;
  random_img.ncol = obs->ncol;
  Generate_Target_Hist_From_Image(ffamily, &random_img);
  random_vector = (float *)malloc(sizeof(float)*ffamily->totalBin);
  
  for (i=0; i < ffamily->nfilter; i++) {
    filter = &(ffamily->fbank[i]);
    for (j=0; j < filter->bin_num; j++) {
      random_vector[ffamily->filterDim[i]+j] = filter->target_hist[j];
    }
  }
  if (k < adatabase->nimages) {
    printf("Observed image is the %dth image in the database.\n",
	   k);
    printf("Loading target histogram from the database.\n");
    for (i=0; i < ffamily->nfilter; i++) {
      filter = &(ffamily->fbank[i]);
      for (j=0; j < filter->bin_num; j++) 
	filter->target_hist[j] = 
	  adatabase->entries[k].fvector[ffamily->filterDim[i]+j];
    }
  }
  else {
    Generate_Target_Hist_From_Image(ffamily, obs);
  }
  fvector = (float *)malloc(sizeof(float)*ffamily->totalBin);
  orderList = (int *)malloc(sizeof(int)*adatabase->nimages);
  distDb = (float *)malloc(sizeof(float)*adatabase->nimages);
  distRandom = (float *)malloc(sizeof(float)*adatabase->nimages);
  distMax = (float *)malloc(sizeof(float)*adatabase->nimages);
  for (i=0; i < ffamily->nfilter; i++) {
    filter = &(ffamily->fbank[i]);
    for (j=0; j < filter->bin_num; j++) {
      fvector[ffamily->filterDim[i]+j] = filter->target_hist[j];
    }
  }

 
  for (k=0; k < adatabase->nimages; k++) {
    maxDist = 0.0;
    distDb[k] = 0.0;
    distRandom[k] = 0.0;
    for (i=0; i < ffamily->totalBin; i++) {
      dist = fabs(fvector[i]-adatabase->entries[k].fvector[i])/2.0;
      distDb[k] += dist;
      if (maxDist < dist) {
	maxDist = dist;
      }
      distRandom[k] += 
	fabs(random_vector[i]-adatabase->entries[k].fvector[i])/2.0;
    }
    distMax[k] = maxDist;
  }
  
  for (k=0; k < adatabase->nimages; k++) {
    orderList[k] = k;
  }
  do {
    i = 0;
    for (k=0; k < (adatabase->nimages-1); k++) {
      if (distDb[orderList[k]] > distDb[orderList[k+1]]) {
	j = orderList[k];
	orderList[k] = orderList[k+1];
	orderList[k+1] = j;
	i = 1;
      }
    }
  } while(i);

  
  for (k=0; k < adatabase->nimages; k++) {
    printf("\t%2d %8.6f %8.6f %s.\n",
	   orderList[k], 
	   distDb[orderList[k]],
	   distMax[orderList[k]],
	   &(adatabase->entries[orderList[k]].fname[adatabase->startIndex]));
  }
  k =0 ;
  for (k=0; k < adatabase->nimages; k++) {
    if (distMax[orderList[k]] > controlInfo->model_too_close)
      break;
  }
  if ( (k+controlInfo->num_ref_model) > adatabase->nimages) {
    controlInfo->num_ref_model = adatabase->nimages-k;
    fprintf(stderr, "Not enough reference model. Was set to %d.\n",
	    controlInfo->num_ref_model);
    if (controlInfo->num_ref_model <= 0) {
      exit(-1);
    }
  }
  
  adatabase->num_ref_model = controlInfo->num_ref_model;
  adatabase->selected_models = (int *)malloc(sizeof(int)*
					     adatabase->num_ref_model);
  dist = 0.0;
  for (i=0; i < ffamily->totalBin; i++) {
    dist += fabs(fvector[i]-random_vector[i])/2.0;
  }
  printf("\n          No.     Ref  Dist-L1     Max     Random-L1     ");
  printf("Model-name\n");
  printf("    Given     ->       %8.6f  %8.6f  %8.6f\n",
	 (float)0.0, (float)0.0, dist);
  
  if ( controlInfo->chosen_manually == 0) {
    
    for (i= 0; i < controlInfo->num_ref_model; i++) {
      j = orderList[i+k];
      adatabase->selected_models[i] = j;
      printf("    Model %2d  ->  %2d   %8.6f  %8.6f  %8.6f   %s.\n",
	     i+1, j, 
	     distDb[j],
	     distMax[j],
	     distRandom[j],
	     &(adatabase->entries[j].fname[adatabase->startIndex]));
    }
  }
  else {
    printf("Here models are chosen manually.\n");
    for (i= 0; i < controlInfo->num_ref_model; i++) {
      j = controlInfo->chosen_ref_models[i];
      adatabase->selected_models[i] = j;
      printf("    Model %2d  ->  %2d   %8.6f  %8.6f  %8.6f   %s.\n",
	     i+1, j, 
	     distDb[j],
	     distMax[j],
	     distRandom[j],
	     &(adatabase->entries[j].fname[adatabase->startIndex]));
    }
  }
  printf("Reading lambda's for selected models.\n");
  for (i=0; i < adatabase->num_ref_model; i++) {
    printf("\tModel %d .... ",adatabase->selected_models[i]);
    j = adatabase->selected_models[i];
    fflush(stdout);
    Read_Lambdas_Into_Vector(adatabase->entries[j].lambdaFile,
			     ffamily,
			     adatabase->entries[j].lambda_vec);
    printf("Done.\n");
  }
  
  free(fvector);
  free(distDb);
  free(orderList);
  free(distMax);
  free(distRandom);
  Free_Image_Int(&random_img);
  return;
}

void Initialize_Obs_Pseudo(FILTERBANK *ffamily, 
			   IMAGEINT *obs,
			   CONTROL_INFO *controlInfo,
			   IMAGE_DATABASE *adatabase)
{
  int i,j,k;
  FILTER *filter;
  
  if (Read_An_Obs_Image(controlInfo->inputFile,
			obs,
			controlInfo) !=0) {
    fprintf(stderr,"Could not load image '%s'.\n",
	    controlInfo->inputFile);
    exit(-1);
  }
  ffamily->norm = 1.0;
  Generate_Target_Hist_From_Image(ffamily, obs); 
  if (controlInfo->num_ref_model == 0) {
    return; 
  }
  if (controlInfo->num_ref_model != 1) {
    printf("For pseudo likelihood, the number of reference models is 1.\n");
    controlInfo->num_ref_model = 1;
  }
  
  adatabase->num_ref_model = controlInfo->num_ref_model;
  adatabase->selected_models = (int *)malloc(sizeof(int)*
					     adatabase->num_ref_model);
  for (i= 0; i < controlInfo->num_ref_model; i++) {
    adatabase->selected_models[i] = 0;
    j = adatabase->selected_models[i];
    for (k=0; k < ffamily->totalBin; k++) {
      adatabase->entries[j].lambda_vec[k] = 0.0;
    }
  }
  return;
}

void Initialize_Syn_Satellite(FILTERBANK *ffamily, IMAGEINT *syn,
			      CONTROL_INFO *controlInfo)
{
  int   k;
  ffamily->greyLevel = controlInfo->greyLevel;
  if (Read_An_Obs_Image(controlInfo->inputFile,
			syn, controlInfo) !=0) {
    fprintf(stderr,"Cannot load image from '%s'.\n",
	    controlInfo->inputFile);
    exit(-1);
  }
  if (controlInfo->num_ref_model==0) {
    printf("Initiliazing pseudo version, where synthesized is loaded from");
  }
  else {
    printf("Initiliazing satellite version, where synthesized is loaded from");
  }
  printf("\n\t%s\n",controlInfo->inputFile);
  ffamily->norm = 1.0;
  printf("Convolving with synthesized image ...");
  fflush(stdout);
  for (k=0; k < ffamily->nfilter; k++) {
    printf(" %d", k);
    fflush(stdout);
    ffamily->fbank[k].patch = Create_Matrix(ffamily->greyLevel,
					    ffamily->fbank[k].size);
    ffamily->fbank[k].patch_hist = Create_Matrix(ffamily->greyLevel,
						 ffamily->fbank[k].bin_num);
    
    Create_An_Image_Float(syn->nrow, syn->ncol,
			  &(ffamily->fbank[k].map));
    Convolution(syn, &(ffamily->fbank[k].map),
		&(ffamily->fbank[k]));       /* fill map[][] */
  }
  printf(" Done!\n");
  
  return;
}


void Re_Initialize_Syn(FILTERBANK *ffamily, IMAGEINT *syn,
		    CONTROL_INFO *controlInfo)
{
  int   k;
  
  Free_Image_Int(syn);
  k = 1;
  if ( strcmp(controlInfo->initialSyntype,"n") &&
       strcmp(controlInfo->initialSyntype,"c")) {
    k = Read_An_Obs_Image(controlInfo->initialSyntype,
			  syn, controlInfo);
    if (k!=0) {
      fprintf(stderr,"Cannot load image from '%s'.\n",
	      controlInfo->initialSyntype);
      fprintf(stderr,"A white noise image will be used instead.\n");
    }
  }
  if (k !=0) {
    if (strcmp(controlInfo->initialSyntype,"c") ==0) {
      if (controlInfo->initialSynval < 0)
	controlInfo->initialSynval = 0;
      else {
	if (controlInfo->initialSynval >= controlInfo->greyLevel)
	  controlInfo->initialSynval = controlInfo->greyLevel-1;
      }
      syn->data = Create_Const_Matrix(controlInfo->syn_height,
				     controlInfo->syn_width,
				     controlInfo->initialSynval);
    }
    else {
      syn->data = Create_Noise_Matrix(controlInfo->syn_height,
				      controlInfo->syn_width,
				      (int)0,
				      controlInfo->greyLevel-1);
    }
    syn->nrow = controlInfo->syn_height;
    syn->ncol = controlInfo->syn_width;
  }
  
  printf("Re-Convolving with synthesized image ...");
  fflush(stdout);
  for (k=0; k < ffamily->nfilter; k++) {
    printf(" %d", k);
    fflush(stdout);
    /*
      ffamily->fbank[k].patch = Create_Matrix(ffamily->greyLevel,
                                 ffamily->fbank[k].size);
      ffamily->fbank[k].patch_hist = Create_Matrix(controlInfo->Grey_Level,
                                       filter->bin_num);
    */
    /* Do not need to re-create the map images even though the program
       should work with it */
    /*
      Create_An_Image_Float(syn->nrow, syn->ncol,
      &(ffamily->fbank[k].map));
    */
    Convolution(syn, &(ffamily->fbank[k].map),
		&(ffamily->fbank[k]));       /* fill map[][] */
  }
  printf(" Done!\n");
  printf("Re-Generating histogram of synthesized image ... ");
  fflush(stdout);
  Generate_Hist_From_Map(ffamily);
  printf(" Done!\n");
  return;
}

void Initialize_Syn(FILTERBANK *ffamily, IMAGEINT *syn,
		    CONTROL_INFO *controlInfo)
{
  int   k;
  ffamily->greyLevel = controlInfo->greyLevel;
  k = 1;
  if ( strcmp(controlInfo->initialSyntype,"n") &&
       strcmp(controlInfo->initialSyntype,"c")) {
    k = Read_An_Obs_Image(controlInfo->initialSyntype,
			  syn, controlInfo);
    if (k!=0) {
      fprintf(stderr,"Cannot load image from '%s'.\n",
	      controlInfo->initialSyntype);
      fprintf(stderr,"A white noise image will be used instead.\n");
    }
  }
  if (k !=0) {
    if (strcmp(controlInfo->initialSyntype,"c") ==0) {
      if (controlInfo->initialSynval < 0)
	controlInfo->initialSynval = 0;
      else {
	if (controlInfo->initialSynval >= controlInfo->greyLevel)
	  controlInfo->initialSynval = controlInfo->greyLevel-1;
      }
      syn->data = Create_Const_Matrix(controlInfo->syn_height,
				     controlInfo->syn_width,
				     controlInfo->initialSynval);
    }
    else {
      syn->data = Create_Noise_Matrix(controlInfo->syn_height,
				      controlInfo->syn_width,
				      (int)0,
				      controlInfo->greyLevel-1);
    }
    syn->nrow = controlInfo->syn_height;
    syn->ncol = controlInfo->syn_width;
  }
  ffamily->norm = (float)(syn->nrow*syn->ncol);
  printf("Convolving with synthesized image ...");
  fflush(stdout);
  for (k=0; k < ffamily->nfilter; k++) {
    printf(" %d", k);
    fflush(stdout);
    ffamily->fbank[k].patch = 
      Create_Matrix(ffamily->greyLevel,
		    ffamily->fbank[k].size);
    ffamily->fbank[k].patch_hist = 
      Create_Matrix(ffamily->greyLevel,
		    ffamily->fbank[k].bin_num);
    Create_An_Image_Float(syn->nrow, syn->ncol,
			  &(ffamily->fbank[k].map));
    Convolution(syn, &(ffamily->fbank[k].map),
		&(ffamily->fbank[k]));       /* fill map[][] */
  }
  printf(" Done!\n");
  printf("Generating histogram of synthesized image ... ");
  fflush(stdout);
  Generate_Hist_From_Map(ffamily);
  printf(" Done!\n");
  return;
}


void Update_Syn_Filters(FILTERBANK *ffamily, IMAGEINT *syn)
{
  int   k;
  for (k=0; k < ffamily->nfilter; k++) {
    if (ffamily->fbank[k].flag ==0) continue;
    Convolution(syn, &(ffamily->fbank[k].map),
		&(ffamily->fbank[k])); 
  }
}


/******************************************************
    Convolve matrix mat[][] with filter->kernel, and
    the results are stored in filter->map_s, _c
*******************************************************/
void Convolution(IMAGEINT *animage, IMAGEFLOAT *mat, FILTER *filter)
{
  MAT    map;
  ARRAY  kernel;
  float  sum;
  int i, r, c, width,height, *pr, *pc;
  int  crow, ccol;
  if ( (mat->nrow != animage->nrow) ||
       (mat->ncol != animage->ncol) ) {
    Create_An_Image_Float(animage->nrow, animage->ncol,
			  mat);
  }
  width=filter->wid; height=filter->ht;
  
  map=filter->map.data; kernel=filter->kernel;
  
  for(r = 0; r< animage->nrow; r++) {
    for(c = 0; c< animage->ncol; c++) {
      sum=0;     
      pr=filter->rlist; pc=filter->clist;
      for(i=0; i<filter->size; i++) { 
	crow = (r+(*pr)+animage->nrow*1000)%animage->nrow;
	ccol = (c+(*pc)+animage->ncol*1000)%animage->ncol;
	sum+=kernel[i]*animage->data[crow][ccol];
	pr++; pc++;
      }
      mat->data[r][c]=sum;
    }
  }
  return;
}



void Skip_Comment_Empty(FILE *fp)
{
  char ch;
  do {
    ch = getc(fp);
    if ( (ch != '\n') && (ch != '#') && (ch != ' ') 
	 && (ch != '\t')) {
      ungetc(ch,fp);
      break;
    }
    if (ch == '#') {
      do {
	ch = getc(fp);
      } while (ch != '\n');
    } 
  } while(1);
  return;
}


int Load_Database(char *fname, FILTERBANK *ffamily,
		  IMAGE_DATABASE *adatabase)
{
  FILE *fp;
  int k, i, j, k1;
  
  if ( (fp=fopen(fname,"rb")) == NULL) {
    printf("Cannot open database file '%s'\n", fname);
    return -1;
  }
  printf("Loading database from file '%s' ... ", fname);
  fflush(stdout);
  /* Read comments and empty lines here */
  Skip_Comment_Empty(fp);
  
  fscanf(fp, "%d", &i);
  if (i != ffamily->nfilter) {
    fprintf(stderr, "Number of filters (%d) in filter mark file does ", 
	    i);
    fprintf(stderr, "not match \nthe current number of filters (%d).\n",
	    ffamily->nfilter);
    fclose(fp);
    return -1;
  }
  for (i=0; i<ffamily->nfilter; i++) {
    fscanf(fp, "%d%d%f", &j, &k1, &(ffamily->fbank[i].unit));
    if (i != j) {
      fprintf(stderr,"Wrong bin mark file format. ");
      fprintf(stderr,"Filter %d starts with %d instead.\n", i, j);
      fclose(fp);
      return -1;
    }
    if (k1 != ffamily->fbank[i].bin_num) {
      fprintf(stderr,"The number of bins (%d) of filter %d ",
	      k1, i);
      fprintf(stderr,"does not match the given filters %d.\n",
	      ffamily->fbank[i].bin_num);
      fclose(fp);
      return -1;
    }
    for (k=0; k < ffamily->fbank[i].bin_num; k++) 
      fscanf(fp, "%f", &(ffamily->fbank[i].mark[k]));
  }
  
  Skip_Comment_Empty(fp);
  
  fscanf(fp,"%d", &(adatabase->nimages));
 
  adatabase->entries = (ONE_IMAGE_ENTRY *)malloc(sizeof(ONE_IMAGE_ENTRY)*
						 adatabase->nimages);
  for (k=0; k < adatabase->nimages; k++) {
    adatabase->entries[k].fvector = Create_Array(ffamily->totalBin);
    adatabase->entries[k].lambda_vec = Create_Array(ffamily->totalBin);
  }
  for (k=0; k < adatabase->nimages; k++) {
    fscanf(fp,"%d", &i);
    if (i != k) {
      printf("\nFormat error when reading data entry %d. ", k);
      printf("It started with %d instead.\n", i);
      fclose(fp);
      return -1;
    }
    fscanf(fp,"%s", adatabase->entries[k].fname);
    
    fscanf(fp,"%d", &i);
    if (i != ffamily->totalBin) {
      printf("\nThe length of the vector %s does not match ",
	     i);
      printf("the current number %d.\n", ffamily->totalBin);
      fclose(fp);
      return -1;
    }
    fscanf(fp,"%s", adatabase->entries[k].lambdaFile);
    fscanf(fp,"%s", adatabase->entries[k].synImage);

    for (i=0; i < ffamily->totalBin; i++) {
      fscanf(fp,"%f", &(adatabase->entries[k].fvector[i]));
    }
  }
  fclose(fp);
  adatabase->startIndex = strlen(adatabase->entries[0].fname)-1;
  do {
    if (adatabase->entries[0].fname[adatabase->startIndex] == '/') {
      adatabase->startIndex ++;
      break;
    }
    adatabase->startIndex --;
  } while(adatabase->startIndex>0);

  printf("Loaded successfully.\n");
  return 0;
}


  
int Save_Database(char *fname, FILTERBANK *ffamily,
		  IMAGE_DATABASE *adatabase)
{
  FILE *fp;
  int k, i;
  
  if ( (fp=fopen(fname,"wb")) == NULL) {
    printf("Cannot save database to file '%s'\n", fname);
    return -1;
  }
  printf("Saving database to file '%s' ... ", fname);
  fflush(stdout);
  fprintf(fp,"#total number of filters\n");
  fprintf(fp,"#each filter: filter-id bin-number unit ");
  fprintf(fp, "mark[0] mark[1] ...\n");
  fprintf(fp, "%d\n", ffamily->nfilter);
  for (i=0; i<ffamily->nfilter; i++) {
    fprintf(fp, "%d %d %12.10f ", i, 
	    ffamily->fbank[i].bin_num,
	    ffamily->fbank[i].unit);
    for (k=0; k < ffamily->fbank[i].bin_num; k++) 
      fprintf(fp, "%12.10f ", ffamily->fbank[i].mark[k]);
    fprintf(fp,"\n");
  }
  fprintf(fp,"\n");
  fprintf(fp,"#Total number of observed images.\n");
  fprintf(fp,"#Each image: image-id image-name ");
  fprintf(fp, "total-bin histogram-data.\n");
  
  fprintf(fp,"%d\n", adatabase->nimages);
  
  for (k=0; k < adatabase->nimages; k++) {
    fprintf(fp,"%d %s  %d\n", k, 
	    adatabase->entries[k].fname,
	    ffamily->totalBin);
    fprintf(fp,"%s\n", adatabase->entries[k].lambdaFile);
    fprintf(fp,"%s\n", adatabase->entries[k].synImage);
    fprintf(fp,"\t");
    for (i=0; i < ffamily->totalBin; i++) {
      fprintf(fp,"%12.10f ", adatabase->entries[k].fvector[i]);
      if ( ((i+1)&0x7) == 0 && (i+1) < ffamily->totalBin) 
	fprintf(fp,"\n\t");
    }
    fprintf(fp,"\n");
  }
  fprintf(fp,"\n");
  fprintf(fp,"#End of the database.\n");
  printf("Saved sucessfully.\n");
  return 0;
}

  
int Create_Database(CONTROL_INFO *controlInfo, 
		    FILTERBANK *ffamily,
		    IMAGE_DATABASE *adatabase)
{
  int k,k1;
  int i,j;
  IMAGEINT inputImg;
  float min, max;
  FILTER *filter;
  inputImg.nrow = 0;
  strcpy(adatabase->hist_Mark, controlInfo->histMark);
  adatabase->num_ref_model =0;
  for (k=0; k < ffamily->nfilter; k++) {
    ffamily->fbank[k].sum_min = 0.0;
    ffamily->fbank[k].sum_max = 0.0;
    if ( (ffamily->fbank[k].wid==0) && (ffamily->fbank[k].ht==0) ) {
      ffamily->fbank[k].sum_min = 256.;
      ffamily->fbank[k].sum_max = 0.0;
    }
  }
  adatabase->segWin = controlInfo->segWin;
  adatabase->topWin = controlInfo->seedWin;
  strcpy(adatabase->hist_Mark, controlInfo->histMark);
  
  for (k=0; k < controlInfo->nInput; k++) {
    for (k1=strlen(controlInfo->inputFiles[k])-1; k1>=0; k1--) {
      if (controlInfo->inputFiles[k][k1] == '/') {
	k1++;
	break;
      }
    }
    if (k1 < 0) k1=0;
    printf("Calculating marks from %s ... \n", 
	   &(controlInfo->inputFiles[k][k1]));
    fflush(stdout);
    Read_An_Obs_Image(controlInfo->inputFiles[k], &inputImg,
		      controlInfo);
    Generate_Marks_From_Image(ffamily, &inputImg, controlInfo);
  }
  for (k=0; k < ffamily->nfilter; k++) {
    filter = &(ffamily->fbank[k]);
    if( (filter->wid==0) && (filter->ht==0) ) { 
      min = (ffamily->fbank[k].sum_min); 
      max = (ffamily->fbank[k].sum_max);
      filter->unit=(float)(max-min+1)/(float)filter->bin_num;
      for(i=0; i<filter->bin_num; i++)
	filter->mark[i]= ((float)i)*filter->unit+min;
      
    }
    else {
      min = (ffamily->fbank[k].sum_min)/(float)controlInfo->nInput;
      max = (ffamily->fbank[k].sum_max)/(float)controlInfo->nInput;
      filter->unit = (max-min)/((float)filter->bin_num);
      for(i=0; i<filter->bin_num; i++)
	filter->mark[i]=((float)i)*filter->unit +
	  min;
      
    }
  }
  
  if (adatabase->nimages >0) {
    Free_Database(adatabase);
  }
  adatabase->nimages = controlInfo->nInput;
  
  adatabase->entries = (ONE_IMAGE_ENTRY *)malloc(sizeof(ONE_IMAGE_ENTRY)*
						 adatabase->nimages);
  
  for (k=0; k < adatabase->nimages; k++) {
    strcpy(adatabase->entries[k].fname, 
	   controlInfo->inputFiles[k]);
    strcpy(adatabase->entries[k].lambdaFile,
	   controlInfo->lambdaFiles[k]);
    strcpy(adatabase->entries[k].synImage,
	   controlInfo->synFiles[k]);
    adatabase->entries[k].fvector = Create_Array(ffamily->totalBin);
    adatabase->entries[k].lambda_vec = Create_Array(ffamily->totalBin);
  }
  
  adatabase->startIndex = strlen(adatabase->entries[0].fname)-1;
  do {
    if (adatabase->entries[0].fname[adatabase->startIndex] == '/') {
      adatabase->startIndex ++;
      break;
    }
    adatabase->startIndex --;
  } while(adatabase->startIndex>0);
  
  for (k=0; k < adatabase->nimages; k++) {
    Read_An_Obs_Image(controlInfo->inputFiles[k], &inputImg,
			controlInfo);
    printf("Calculating histograms of %s ... \n", 
	   &(controlInfo->inputFiles[k][adatabase->startIndex]));
    Generate_Target_Hist_From_Image(ffamily, &inputImg);
    /* Save the histogram to the feature vector */
    for (i=0; i < ffamily->nfilter; i++) {
      filter = &(ffamily->fbank[i]);
      for (j=0; j < filter->bin_num; j++) 
	adatabase->entries[k].fvector[ffamily->filterDim[i]+j] = 
	  filter->target_hist[j];
    }
    
  }
  Free_Image_Int(&inputImg);
  return 0;
}


void Free_Database(IMAGE_DATABASE *adatabase)
{
  if (adatabase->nimages > 0) {
    int k;
    for (k=0; k < adatabase->nimages; k++) {
      Free_Array(adatabase->entries[k].fvector);
      Free_Array(adatabase->entries[k].lambda_vec);
    }
    free(adatabase->entries);
    /* if (adatabase->num_ref_model >0) {
      free(adatabase->selected_models);
      adatabase->num_ref_model = 0;
      }*/
    adatabase->nimages = 0;
  }
  return;
}






