#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "type-def.h"
#include "util.h"
#include "kernel.h"
#include "imageio.h"
#include "smoothing.h"
#include "para-util.h"
#include "hist-util.h"
#include "filter-util.h"
#include "julesz-util.h"

int main(int argc, char **argv)
{
  char paraFile[256];
  IMAGE_DATABASE imageDataBase;
  int windex, upindex;
  int wsize;
  char prefix[256];
  char fname[256];
  int i,k;
  CONTROL_INFO  para_setting;
  FILTERBANK  fbank;
  IMAGEINT   obs_img, syn_img;
  float  total_err, max_err, weighted_err;
  float total_err_sum;
  int   total_err_times;
  FILE *errfp;

  obs_img.nrow = 0;
  syn_img.nrow = 0;
  if (argc < 1) {
    printf("Please specify the parameter file: ");
    scanf("%s", paraFile);
  }
  else {
    strcpy(paraFile, argv[1]);
  }
  
  LoadParameter(paraFile, &para_setting);
  DisplayParameter(&para_setting);
  
  srand(para_setting.seed);

  fbank.nfilter = 0;

  imageDataBase.nimages =0;
  Read_Filters(para_setting.filterFile, &fbank);
  
  
  if (Load_Database(para_setting.databaseFile, &fbank,
		    &imageDataBase) != 0) {
    Create_Database(&para_setting, &fbank,
		    &imageDataBase);
    Save_Database(para_setting.databaseFile, &fbank,
		  &imageDataBase);
  }
  
  Initialize_Obs_Analytical(&fbank, &obs_img,
			    &para_setting);
  sprintf(fname,"%s_obs.pgm", para_setting.prefix);
  Write_An_Image_Int(&obs_img, (int)(256/para_setting.greyLevel),
		     fname);
  Initialize_Syn(&fbank, &syn_img, &para_setting);

  Scale_Target_Hist(&fbank, (float)(fbank.norm));

  sprintf(fname,"%s_syn_init.pgm", para_setting.prefix);
  Write_An_Image_Int(&syn_img, (int)(256/para_setting.greyLevel),
		     fname);
  
  /* Force not to change lambda's */
  para_setting.learningAlpha = 0.0;
  Update_Syn_Filters(&fbank, &syn_img);
  Generate_Hist_From_Map(&fbank);
  Calc_Julesz_Temp_BasedOn_Entropy(&syn_img, &fbank, &para_setting);
  /* Copy the parameter valuse for julesz */

  para_setting.synInitTemp = para_setting.julesz_synInitTemp;
  para_setting.synCoolingRatio =  para_setting.julesz_synCoolingRatio;
  para_setting.synLowTemp = para_setting.julesz_synLowTemp;
  para_setting.synTemp = para_setting.synInitTemp;
  for (i=0; i < para_setting.synStartIter; i++) {
    para_setting.synTemp /= para_setting.synCoolingRatio;
    if (para_setting.synTemp < para_setting.synLowTemp) 
      para_setting.synTemp = para_setting.synLowTemp;
  }
  
  total_err_sum = 0.0;
  total_err_times = 0;
  for (i= para_setting.synStartIter; i <= para_setting.synIter; i++) {  
    if ( (i%para_setting.synErrIter) ==0) {
      Update_Syn_Filters(&fbank, &syn_img);
      Generate_Hist_From_Map(&fbank);
      printf("Calculating errors ... \n");
      sprintf(fname,"%s_julesz_syn_Histogram.dat", para_setting.prefix);
      total_err = Update_Lambda(i,fname, 
				&fbank,
				&para_setting,
				&max_err,
				&weighted_err);
      if (i>= para_setting.total_Err_avg_iter) {
	total_err_sum += total_err;
	total_err_times++;
      }
      
      sprintf(fname,"%s_julesz_syn_Err.dat", para_setting.prefix);
      if (i==0) {
	errfp = fopen(fname,"wb");
      }
      else {
	errfp = fopen(fname,"ab");
      }
      fprintf(errfp,"%d with max_error = %8.6f ", i, max_err);
      fprintf(errfp,"weighted error: ");
      fprintf(errfp,"%8.6f ", weighted_err);
      fprintf(errfp,"Total = %8.6f \n", total_err);
      fclose(errfp);
      
      if (max_err < para_setting.synMaxErrEpsilon) break;
    }
    if ( (i%para_setting.saveSynIter) ==0) {
      sprintf(fname,"%s_julesz_syn_%d.pgm", para_setting.prefix,i);
      Write_An_Image_Int(&syn_img, (int)(256/para_setting.greyLevel),
			 fname);
    }
    if (Flip_Julesz(&syn_img, &fbank, &para_setting)==0) break;
  }
  if (total_err_times >0) {
    sprintf(fname,"%s_julesz_syn_Err.dat", para_setting.prefix);
    errfp = fopen(fname,"ab");
    fprintf(errfp,"\n\n#Total times: %d average: %f\n",
	    total_err_times, (float)(total_err_sum/total_err_times));
    fclose(errfp);
  }
  sprintf(fname,"%s_julesz_syn_final_%d.pgm", para_setting.prefix,i);
  Write_An_Image_Int(&syn_img, (int)(256/para_setting.greyLevel),
		     fname);
  Free_Image_Int(&obs_img);
  Free_Image_Int(&syn_img);
  Free_Filter_Bank(&fbank);
  Free_Database(&imageDataBase);
  return 0;
}














