#' Log-likelihood
#' 
#' Computes Gaussian log-likelihood based on an identity
#' covariance matrix. Internal function for
#' solsel.
#' 
#' @param yi a numeric vector representing a 
#' data point.
#' @param mu a numeric matrix of cluster means.
#' Each row of the matrix represents a cluster mean.
#' @param propk a numeric vector of proportions of 
#' data points in each cluster.
#' @return Returns log-likelihood for a data point yi.
loglik <- function(yi, mu, propk) {
  p <- ncol(mu)
  normc <- sapply(1:nrow(mu), function(x) ((2*pi)^(-p/2))^(-1/2))
	SS <- sapply(1:nrow(mu), function(x) crossprod(yi - mu[x, ]))
	normd <- propk*normc*exp(-1/2*SS)
	li <- log(sum(normd))
	return(sum(li))	
}

#' Solution selection
#' 
#' @param y a numeric matrix of data.
#' @param assignV a list of numeric vectors of cluster 
#' assignments for a solution path.
#' @param a value in [0, 1] for the difference ratio cut off.
#' Defaults to 0.05. Recommended at (0.05, 0.1).
#' @return \item{bestsol}{ the number of the best solution 
#' selected from a solution path.}
#' \item{bestK}{ the estimated number of cluster selected.}
#' \item{diffratio}{a vector of difference ratios.}
#' 
#' @references Marchetti, Y., Zhou, Q. Solution path clustering with adaptive concave penalty. 
#' Electron. J. Statist. 8 (2014), no. 1, 1569--1603.
#' 
#' @examples
#' 
#' # simulated a dataset with 10 well-separated clusters
#' sim <- simclust(K=10, p=20, ncl=400, noise=0, overnk=0, hcube=c(-5, 5), random=5248)
#' 
#' # cluster with solution path clustering
#' fitspc <- spc(y=sim$y, omega=0.5, out=FALSE)
#' 
#' # select a solution and compute ARI scores for the selected solution
#' ss <- solsel(y=sim$y, assignV=fitspc$assignV, a=0.05)
#' ARI(assignV=fitspc$assignV[[ss$bestsol]], k0=3, assignT1=sim$assignT1)
#' 
#' @export
solsel <- function(y, assignV, a=0.05) {
	n <- nrow(y)
  p <- ncol(y)
	assignDR <- vector( "list", 1 )
	assignDR[[ 1 ]] <- 1:n
	assignDR <- c(assignDR, assignV)
	Lj <- c()
	Ksj <- c()
	for( j in 1:length(assignDR)) {
		class <- assignDR[[ j ]]
		size <- sapply(unique(class), function(x) length(which(class == x)))
		mu <- t(sapply(unique(class), function(x) meanclust(y=y, x=x, class=class, 
                                                        trimprop=0, center="mean")))
		propk <- size / n
		Ksj <- c(Ksj, length(unique(class)))
		li <- apply(y, 1, function(x) loglik(yi=x, mu=mu, propk=propk))
		Lj <- c(Lj, sum(li))
		}
  
  diffratio <- diff(rev(Lj)) / diff(rev(Ksj))
  cutoff <- max(which(diffratio > max(diffratio)*a))
	bestsol <- length(assignV) - cutoff
  bestK <- rev(Ksj)[cutoff + 1]
  
  return(list(bestsol=bestsol, bestK=bestK, diffratio=diffratio))
  }


